/** @file   animframe.h
 * @brief   Declaration of AnimFrame - class.
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:37 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_ANIMFRAME_H
#define H_ENG2D_ANIMFRAME_H

#include <iostream>
#include "eng2d_dll.h"
#include "datatypes.h"

// Forward declarations
struct BITMAP;
struct RLE_SPRITE;

namespace eng2d {


/** @class  AnimFrame
 * @brief   AnimFrame represents a single frame in an animation.
 * @author  Tomi Lamminsaari
 */
class DLLIMPORT AnimFrame
{
public:

  ///
  /// Members
  /// =======
  
  /** The ID-number of the graphics that should be shown when this frame
   * is visible. Usually this operates as an index to the bitmap-table that
   * has all the graphics that is needed for the animation.
   */
  int         iFrameIndex;
  
  /** Duration of this frame. How many update-calls this frame is visible. */
  int         iFrameDelay;
  
  /** Index of the sound that should be played when this frame is shown. -1
   * means no sound.
   */
  int         iFrameSound;
  
  /** The blending mode being used for this frame. */
  BlendMode   iFrameBlend;
  
  /** Alpha value */
  int         iFrameAlpha;
  

  ///
  /// Constructors, destructor and operators
  /// ======================================

  /** Constructs new AnimFrame. The 'frameIndex' member will be set to -1.
   */
  AnimFrame();
  
  /** Constructs new AnimFrame.
   * @param     aIndex            The frameIndex-data.
   * @param     aDelay            The duration of this frame.
   */
  AnimFrame( int aIndex, int aDelay );
  
  /** Destructor.
   */
  virtual ~AnimFrame();
  


  ///
  /// Public methods
  /// ==============

  /** Sets the graphics pointer.
   * @param   aType             Type of the graphics at the pointer.
   * @param   aGfx              Pointer to graphics that relates to this
   *                            frame. Ownership is not transferred.
   */
  void setGraphics(GfxContentType aType, void* aGfx);
  
  /** Reads the frame data from given stream. This method expects that the
   * opening F-tag has been read away from the stream.
   * @param   aIn               The input stream to read from.
   * @return  KErrNone if successful.
   */
  int read( std::istream& aIn );



  ///
  /// Getter methods
  /// ==============
  
  /** Returns the graphics pointer as Allegro BITMAP.
   * @return  Pointer to Allegro BITMAP. If content type is not EBitmap, a
   *          nullpointer will be returned.
   */
  BITMAP* asBitmap() const;
  
  /** Returns the graphics pointer to Allegro RLE_SPRITE.
   * @return  Pointer to Allegro RLE_SPRITE. If content type is not ERleSprite
   *          a nullpointer will be returned.
   */
  RLE_SPRITE* asRleSprite() const;
  
  /** Returns the type of the graphics pointer.
   * @return  Type of the graphics in iGfx - pointer.
   */
  GfxContentType gfxType() const;
  
  
protected:
  ///
  /// Members
  /// =======
  
  /** A member where you can place a pointer to any graphics that is related
   * to this frame.
   */
  void*       iGfx;
  
  /** Type of the graphics in @c iGfx pointer. */
  GfxContentType  iGfxType;
};


};

#endif
